//*****************************************************************************
//      Filename : ugForceField.h
// ----------------------------------------------------------------------------
// This file is used to detect collision and generate ForceField
//
// By Laehyun Kim
// University of Southern California
//
//*****************************************************************************


#ifndef UGForceField_INCLUDE
#define UGForceField_INCLUDE


#include <gstForceField.h>

#include "global.h"

// ForceField derived class that represents a sphere with magnetic
// attraction.
class ugForceField : public gstForceField 
{
public:
	
    // Constructor.
    ugForceField(double sphereRadius, // radius of sphere
                   double fieldRadius); // radius of attracting field

	~ugForceField() { }


    // Standard GHOST SDK node methods
    // Must be defined for all derived classes.

    // Get type of this class.  No instance needed.
    static gstType getClassTypeId()
    {
        gstAssignUniqueId(ugForceFieldClassTypeId);
	return ugForceFieldClassTypeId;
    }

    // Get type of this instance.
    virtual gstType getTypeId() const { return getClassTypeId(); }

    // Returns TRUE if this class is same or derived class of type.
    virtual gstBoolean isOfType(gstType type) const {return staticIsOfType(type);}

    // Returns TRUE if subclass is of type.
    static gstBoolean staticIsOfType(gstType type)
    {
        if (type == getClassTypeId()) return TRUE;
	else return (gstForceField::staticIsOfType(type));
    }

    // gstForceField method overloaded to define the force applied by
    // the PHANToM when the  PHANToM has entered the bounding volume
    // associated with the force field
    virtual gstVector calculateForceFieldForce(gstPHANToM *phantom);
    virtual gstVector calculateForceFieldForce(gstPHANToM *phantom,
                                               gstVector &torque);

	bool    isContact() { 
			if( inContact ) return true;
			else return false;
	}
	void    getTooltipPoint( float &tx, float &ty, float &tz ) {
			tx = tooltipPos[0];
			ty = tooltipPos[1];
			tz = tooltipPos[2];
	}
	void    getContactPoint( float &cx, float &cy, float &cz ) {
			cx = surfacePos[0];
			cy = surfacePos[1];
			cz = surfacePos[2];
	}
	void    getForceVector( float &fx, float &fy, float &fz ) {
			fx = surfacePos[0] - globalPos[0];
			fy = surfacePos[1] - globalPos[1];
			fz = surfacePos[2] - globalPos[2];
	}

	float   getFrictionValue() { return friction_value; }
	void    setFrictionValue( float fv ) { 
				if( fv <= 0.0 ) {
					friction_on = false;
					friction_value = 0.0;
				}
				else {
					friction_on = true;
					if( fv >= 1.0 ) friction_value = 1.0;
					else friction_value = fv;
				}
			}

	float   getSurfacePotential() { return surfacePotential; }
	void    setSurfacePotential(float sp) { surfacePotential = sp; }

	float   getStiffness() { return stiffness; }
	void    setStiffness(float sn) { 
				if( sn <=  0.3 ) stiffness = 0.3;
				else if( sn >= 1.0 ) stiffness = 1.0;
				else stiffness = sn;
			}

	bool	recoverableError() {
				if( isError && canRecover ) return true;
				else return false;
			}

	void    resetErrorValable() {
				canRecover = false;
				isError = false;
			}

	float	globalPos[3];
	float   surfacePos[3];
	float   tooltipPos[3];

	float   currentX, currentY, currentZ;
	float   currentF, newF;
	float	dt[3];
	bool    inContact;
	float	contactP[3];
	float	preFV[3];  // previous force vector
	float	preCP[3];  // previous contact position

	float   preForce;  // previous norm of force


private:
	static gstType ugForceFieldClassTypeId;

	double msphereRadius, mfieldRadius;

	float  surfacePotential;
	bool   friction_on;
	float  friction_value;
	float  stiffness;

	float  LinInterp3D(float px, float py, float pz);
	void   normalInterp3D(float px, float py, float pz, float &rx, float &ry, float &rz);
	void   findNearestPoint(float cp[3], float rp[3], float pv, float &rx, float &ry, float &rz);
	void   findNearestPoint_2(float cp[3], float cn[3], float pv, float &rx, float &ry, float &rz);

	void   findCP(float p1[3], float p0[3], float nf, float &cp1, float &cp2, float &cp3);
	void   calc_normal(int i, int j, int k, float &nx, float &ny, float &nz);

	void   indexConverter(float ix, float iy, float iz, float &ox, float &oy, float &oz);


};

#endif
