// -*- C++ -*-
//----------------------------------------------------------------------
//
//                            Sean Mauch
//                California Institute of Technology
//                   (C) 2000 All Rights Reserved
//
//----------------------------------------------------------------------

// $Log: cpt.h,v $
// Revision 1.31  2001/01/09 08:51:40  sean
// Added documentation.
//
// Revision 1.30  2001/01/07 22:40:07  sean
// Added Fortran and C bindings for flood fill.
//
// Revision 1.29  2001/01/07 22:33:01  sean
// Added: int cpt_flood_fill_distance( const Real far_away )
//
// Revision 1.28  2001/01/07 03:15:51  sean
// Made HUGE stand for unknown.
//
// Revision 1.27  2000/09/17 08:22:19  sean
// Added the gradient of the distance to the computed arrays.
//
// Revision 1.26  2000/09/05 19:27:21  sean
// Added cpt_valid_grid().
//
// Revision 1.25  2000/08/17 23:51:06  aivazis
// Repaired the FORTRAN bindings
//
// Revision 1.24  2000/07/11 00:20:52  sean
// Updated comments for cpt().
//
// Revision 1.23  2000/06/19 12:37:22  sean
// Changed cpt_init_brep.
//
// Revision 1.22  2000/06/17 21:50:26  sean
// Added cpt_check_mesh()
//
// Revision 1.21  2000/06/16 23:16:59  sean
// Added some documentation.
//
// Revision 1.20  2000/06/16 21:30:02  rapa
// sorry sean, FLT_MAX replaced with more standard HUGE
//
// Revision 1.19  2000/05/18 15:46:21  sean
// Added doxygen style documentation.
//
// Revision 1.18  2000/04/19 17:54:08  sean
// Added cpt_init_brep_clip().
//
// Revision 1.17  2000/04/18 07:38:37  sean
// Now b-rep is made without the cartesian domain.
// Added cpt_write_grid_state() and cpt_check_grid().
//
// Revision 1.16  2000/04/17 11:40:34  sean
// Added brute force algorithm.
//
// Revision 1.15  2000/04/10 20:01:09  sean
// Added cpt_write_mesh().
//
// Revision 1.14  2000/04/10 18:32:13  aivazis
// Added name translation
//
// Revision 1.13  2000/04/10 17:14:52  sean
// changed fortran bindings.
//
// Revision 1.12  2000/04/05 11:28:21  sean
// Changed init procedure.  Added transpose.
//
// Revision 1.11  2000/03/20 16:02:44  sean
// *** empty log message ***
//
// Revision 1.10  2000/03/09 11:10:12  sean
// Took out inside Polyhedron checking to speed up.
//
// Revision 1.9  2000/03/04 22:28:48  sean
// Added closest face ability to cpt_find_sign_distance.
//
// Revision 1.8  2000/02/24 09:03:12  sean
// Added closest face capability.
//
// Revision 1.7  1999/12/01 23:30:26  sean
// Changed to accomidate CPTState class.
// cpt() split into cpt_init() and cpt().
//
// Revision 1.6  1999/11/15 21:34:02  sean
// Added parallel code.
//
// Revision 1.5  1999/11/10 12:01:29  sean
// Added cpt_find_sign_distance
//
// Revision 1.4  1999/11/04 00:59:11  sean
// Changed to Fortran style arrays.
//
// Revision 1.3  1999/11/03 23:36:58  sean
// *** empty log message ***
//
// Revision 1.2  1999/11/03 18:30:31  rapa
// Changed float to Real
// Added a binding for C clients
//

/*! 
  \file cpt.h
  \brief The Closest Point Transform to a triangle surface.
*/

/*!
  \mainpage Closest Point Transform to a Triangle Surface.


  \section introduction Introduction

  This code implements an algorithm for computing the closest point 
  transform to a triangle surface on a regular 3D grid.  The closest point 
  transform finds the closest point, the closest triangle face the 
  Euclidean distance to the triangle mesh and the gradient of this distance
  for the points in a regular grid.  The distance, etc., are computed to
  within a specified distance of the surface.  The closest point, closest
  face and distance and gradient of the distance to the mesh surface 
  are calculated by solving the 
  Eikonal equation \f$ |\nabla u|^2 = 1 \f$ with the method
  of characteristics.  The method of characteristics is implemented efficiently
  with the aid of computational geometry and polyhedron scan conversion.
  The computed distance is accurate to within machine precision.  
  The computational complexity of the algorithm is linear in both the 
  number of grid points for which the distance is computed and the size of 
  the mesh.  Thus it has optimal computational complexity.


  \section compiling Compiling the Library

  The library is composed of the two header files cpt_defs.h and cpt.h
  and the archive cpt.a.  Edit cpt_defs.h to choose options such as 
  single or double precision arithmetic and whether progress information
  is written to stdout.
  
  Under the ASCI build procedure make the library with the "mm" command.
  This executes the makefile "Make.mm".  Otherwise make the library with
  gnu "make".  This executes "Makefile".  Each of these builds the 
  cpt.a archive.


  \section linking Linking with the Library

  Include the header file cpt.h and link with cpt.a.


  \section functions Functions

  cpt() performs the closest point transform.  Before calling cpt(), you
  must call:
  - cpt_init_parameters() to specify how far to compute the distance.  Note 
    that the computational complexity of the alorithm is proportional to 
    this maximum distance.  (The number of grid points for which
    the distance is computed is proportional to the maximum distance.)  
    If performance is an issue, make sure that the \a max_distance
    parameter is no larger than needed.
  - cpt_init_grid() to give the dimensions and location of the grid.  Here
    you specify arrays, (which you must allocate beforehand), to hold the
    distance, closest point and closest face.  Computing the closest point
    and closest face is optional. 
  - cpt_init_brep() to specify the triangular surface.  The triangular 
    surface is a boundary representation, b-rep, of the volume that is 
    inside the surface.  

  After that, you can call any or all of the three initialization functions 
  and then call cpt() to calculate the distance for new parameters, a new 
  grid, or a new b-rep.
  
  If the algorithm is being run on multiple processes, a call to 
  cpt_find_sign_distance() may be needed after each cpt() call to determine 
  the sign of the distance when the distance is unknown.

  There are C and Fortran wrappers for each of the functions.  They have
  a trailing _c and _f, respectively in the function names.


  \section section_driver The Driver 
  
  \ref driver reads a b-rep from a file and writes the distance, 
  the gradient of the distance, the closest point and the closest face 
  transforms to files.

*/

//----------------------------------------------------------------------------

#include "cpt_defs.h"

//---------------------------------------------------------------------------
//                       Initialize the parameters.
//---------------------------------------------------------------------------

/*! 
  \fn void cpt_init_parameters( const Real max_distance )
  \brief Intitialize parameters for the Closest Point Transform.

  C++ entry point.

  This function must be called at least once before calls to cpt().
  
  \param max_distance
  The distance will be computed up to max_distance away from the surface.

  The distance for grid points whose distance is larger than
  max_distance will be set to HUGE.  The closest point of these far away
  points will be set to (HUGE, HUGE, HUGE).  The closest face of far away
  points will be set to -1.
*/
void 
cpt_init_parameters( const Real max_distance );


//---------------------------------------------------------------------------
//                       Initialize the grid.
//---------------------------------------------------------------------------

//! Intitialize grid information for the Closest Point Transform.
/*!
  C++ entry point

  This function must be called at least once before calls to cpt().

  \param cartesian_domain 
  is the domain that the grid spans.
  cartesian_domain[] = {xmin, ymin, zmin, xmax, ymax, zmax}
  \param grid_dimensions 
  is the number of grid points in the \f$ x,y,z \f$ directions
  grid_dimensions[] = {grid_size_x, grid_size_y, grid_size_z}
  \param distance 
  is an array of size 
  \f$ (grid\_dimensions[0] * grid\_dimensions[1] * grid\_dimensions[2]) \f$.
  This array will be assigned the distance from the grid points to the 
  surface by the cpt() function.
  \f$ distance[] = \{ distance(0,0,0), distance(1,0,0), 
  distance(2, 0, 0), \ldots \} \f$
  \param grad_distance
  is an array of size 
  \f$(3 * grid\_dimensions[0] * grid\_dimensions[1] * grid\_dimensions[2])\f$.
  This array will be assigned the gradient of the distance
  by the cpt() function.  It is computed from the geometric primitives and 
  not by differencing the distance array.  Thus it is accurate to machine 
  precision.  If \f$ grad\_distance = 0 \f$ the gradient of the distance
  will not be computed.
  \f$ grad\_distance[] = \{ gd_x(0,0,0), gd_y(0,0,0), gd_z(0,0,0), 
  gd_x(1,0,0), gd_y(1,0,0), gd_z(1,0,0), \ldots \} \f$
  \param closest_point 
  is an array of size 
  \f$(3 * grid\_dimensions[0] * grid\_dimensions[1] * grid\_dimensions[2])\f$.
  This array will be assigned the closest point on the triangle surface 
  by the cpt() function.  If \f$ closest\_point = 0 \f$ the closest points 
  will not be computed.
  \f$ closest\_point[] = \{ cp_x(0,0,0), cp_y(0,0,0), cp_z(0,0,0), 
  cp_x(1,0,0), cp_y(1,0,0), cp_z(1,0,0), \ldots \} \f$
  \param closest_face 
  is an array of size 
  \f$ (grid\_dimensions[0] * grid\_dimensions[1] * grid\_dimensions[2]) \f$.
  This array will be assigned the index of the closest face by the cpt() 
  function.  If \f$ closest\_face = 0 \f$ the closest faces will not be 
  computed.
  \f$ closest\_face[] = \{ cf(0,0,0), cf(1,0,0), cf(2,0,0), \ldots \} \f$
*/
void 
cpt_init_grid( const Real* cartesian_domain, 
	       const int* grid_dimensions,
	       Real* distance,
	       Real* grad_distance,
	       Real* closest_point,
	       int* closest_face );


//---------------------------------------------------------------------------
//                       Initialize the b-rep.
//---------------------------------------------------------------------------


//! Initialize the b-rep.
/*!
  C++ entry point

  Do not use the Cartesian domain to clip the mesh.
  
  Either this function or another cpt_init_brep_() must be called at least 
  once before calls to cpt().

  \param num_vertices 
  is the number of vertices in the surface.
  \param vertices
  is the array containing the cartesian coordinates of the verices in the 
  surface.  \f$ vertices[] = \{ x_0, y_0, z_0, x_1, y_1, z_1, \ldots \} \f$
  \param num_faces 
  is the number of faces in the triangle surface.
  \param faces
  is an array of vertex indices.  Three indices describe a face.
  The indices have positive orientation.
  \f$ faces[] = \{ face_0index_0, face_0index_1, face_0index_2,
  face_1index_0, face_1index_1, face_1index_2, \ldots \} \f$
*/
void 
cpt_init_brep_noclip( const int num_vertices,
		      const Real* vertices,
		      const int num_faces,
		      const int* faces );


//---------------------------------------------------------------------------
//                       Initialize the b-rep.  
//    Clip the mesh to use only points that affect the cartesian domain.
//---------------------------------------------------------------------------

//! Initialize the b-rep.  Clip the mesh.
/*!
  C++ entry point

  Clip the mesh to use only points that affect the cartesian domain.

  Either this function or another cpt_init_brep_() must be called at least 
  once before calls to cpt().

  \param num_vertices 
  is the number of vertices in the surface.
  \param vertices
  is the array containing the cartesian coordinates of the verices in the 
  surface.  \f$ vertices[] = \{ x_0, y_0, z_0, x_1, y_1, z_1, \ldots \} \f$
  \param num_faces 
  is the number of faces in the triangle surface.
  \param faces
  is an array of vertex indices.  Three indices describe a face.
  The indices have positive orientation.
  \f$ faces[] = \{ face_0index_0, face_0index_1, face_0index_2,
  face_1index_0, face_1index_1, face_1index_2, \ldots \} \f$
*/
void 
cpt_init_brep( const int num_vertices,
	       const Real* vertices,
	       const int num_faces,
	       const int* faces );


//---------------------------------------------------------------------------
//                      Closest Point Transform.
//---------------------------------------------------------------------------

//! Compute the Closest Point Transform.
/*!
  C++ entry point

  Compute the distance.  Compute the gradient of the distance, the
  closest face and closest point if their arrays specified in
  cpt_init_grid() are nonzero.

  The algorithm uses polyhedron scan conversion to determine which grid
  points are close to the shapes comprising the surface.

  The unknown distances, gradients, and closest points are set to HUGE
  the unknown closest faces are set to -1.
*/
void 
cpt( void );


//---------------------------------------------------------------------------
//    Closest Point Transform using bbox scan conversion algorithm.
//---------------------------------------------------------------------------

//! Compute the Closest Point Transform.
/*!
  C++ entry point

  The algorithm uses bounding box scan conversion to determine which grid
  points are close to the shapes comprising the surface.  This function 
  performs the same action as cpt(), but is slower.
*/
void 
cpt_bbox( void );


//---------------------------------------------------------------------------
//    Closest Point Transform using brute force algorithm.
//---------------------------------------------------------------------------

//! Compute the Closest Point Transform.
/*!
  C++ entry point

  The algorithm uses a brute force algorithm to compute the distance, 
  closest point and closest face.  This function performs the same action 
  as cpt(), but is much slower.
*/
void 
cpt_brute( void );




//---------------------------------------------------------------------------
//    Flood fill the distance.
//---------------------------------------------------------------------------

//! Flood fill the distance.
/*!
  C++ entry point

  The distance is flood filled.  If any of the distances are known, return
  true and set the unknown distances to +- far_away.  If no distances are 
  known, return false and set all distances to + far_away.
*/
int 
cpt_flood_fill_distance( const Real far_away );




//---------------------------------------------------------------------------
//    Check if the grids are valid.
//---------------------------------------------------------------------------

//! Check the grids.
/*!
  C++ entry point

  Verify that the distance grid is valid.  The known distances should
  be between +-max_distance.  The difference between adjacent grid
  points should not be more than the grid spacing.  Verify that the
  closest point and closest face grids are valid.  Return 1 if the grids
  are valid.  Return 0 and print a message to stderr otherwise.
*/
int 
cpt_valid_grid( void );




//---------------------------------------------------------------------------
//              Find the sign of the distance in the processors.
//---------------------------------------------------------------------------

// CONTINUE: verify perhaps fix

//! Compute the Closest Point Transform.
/*!
  C++ entry point

  If the sign of the distance is unknown: Find the sign of the distance
  using the sign of the distance from the other processes.  This function
  should be called only if cpt() returns 0, i.e. if the sign of the 
  distance is unknown.  If the sign of the distance can be found, all the 
  distances will be set to +- far_away, the closest points will be set to
  ( HUGE, HUGE, HUGE ) and the closest faces will be set to -1.

  \param process_dimensions 
  is an array of the number of processes in the x,y,z directions.
  \f$ process\_dimensions[] = \{ np_x, np_y, np_z \} \f$
  \param process 
  is an array of the coordinates of this process.
  \f$ process[] = \{ p_x, p_y, p_z \} \f$
  \param sign_distance 
  is an array that contains the sign of the distance at the lower and upper 
  corners of each processes domain.
  \f$ sign\_distance = \{ sd\_lower(0,0,0), sd\_lower(1,0,0), 
  sd\_lower(2,0,0), \ldots, sd\_upper(0,0,0), sd\_upper(1,0,0), 
  sd\_upper(2,0,0), \ldots \} \f$

  \return
  A value of 1 is returned if the sign of the distance can be found.
  0 is returned otherwise.
*/
int 
cpt_find_sign_distance( const int* process_dimensions,
			const int* process,
			const int* sign_distance );


//---------------------------------------------------------------------------
//                    Check if a surface is oriented.
//---------------------------------------------------------------------------

//! Check if the surface is oriented.
/*!
  \return
  1 if the surface is oriented, 0 if not.
*/
int 
oriented_surface( const int num_faces, const int* faces );


//---------------------------------------------------------------------------
//                            Orient a surface.
//---------------------------------------------------------------------------

//! Try to make the surface oriented.
void 
orient_surface( const int num_faces, int* faces );


//---------------------------------------------------------------------------
//        Write information about the state of the grid variables.
//---------------------------------------------------------------------------

//! Write information about the state of the grid variables.
void
cpt_write_grid_state( void );


//---------------------------------------------------------------------------
//        Write information about the state of the brep variables.
//---------------------------------------------------------------------------

//! Write information about the state of the b-rep variables.
void
cpt_write_brep_state( void );


//---------------------------------------------------------------------------
//        Write information about the state of the variables.
//---------------------------------------------------------------------------

//! Write information about the state of the variables.
void
cpt_write_state( void );


//---------------------------------------------------------------------------
//          Write information about the state of the grid.
//---------------------------------------------------------------------------

//! Write information about the grid.  How many distances are known.
void
cpt_check_grid( void );


//---------------------------------------------------------------------------
//          Write information about the state of the grid.
//---------------------------------------------------------------------------

//! Write information about the mesh.
void
cpt_check_mesh( int num_vertices, Real* vertices, int num_faces, int* faces );


//---------------------------------------------------------------------------
//                          Write the mesh.
//---------------------------------------------------------------------------

//! Write the mesh in ascii format.
/*!
  Write the mesh to the file "mesh.ascii".  The file format is: 
  \verbatim
  num_vertices  num_faces
  vertex_0_x  vertex_0_y  vertex_0_z
  vertex_1_x  vertex_1_y  vertex_1_z
  ...
  face_0_index_0  face_0_index_1  face_0_index_2
  face_1_index_0  face_1_index_1  face_1_index_2
  ...
  \endverbatim
 */

// commented out by LHK(04/20/2001)
//void cpt_write_mesh( void );


//---------------------------------------------------------------------------
//                           Array transpose.
//---------------------------------------------------------------------------


//! Transpose an array of integers.  Utility function.
/*!
  \pre
  \f$ array = \{ a_{00}, a_{01}, a_{02}, ..., 
  a_{10}, a_{11}, a_{12}, ... \} \f$
  \post
  \f$ array = \{ a_{00}, a_{10}, a_{20}, ..., 
  a_{01}, a_{11}, a_{21}, ... \} \f$
*/
void
cpt_transpose_int( int xdim, int ydim, int* array );

//! Transpose an array of reals.  Utility function.
/*!
  \pre
  \f$ array = \{ a_{00}, a_{01}, a_{02}, ..., 
  a_{10}, a_{11}, a_{12}, ... \} \f$
  \post
  \f$ array = \{ a_{00}, a_{10}, a_{20}, ..., 
  a_{01}, a_{11}, a_{21}, ... \} \f$
*/
void
cpt_transpose_real( int xdim, int ydim, Real* array );


//===========================================================================
//===========================================================================
//                        C and Fortran Interface
//===========================================================================
//===========================================================================

#if defined(__cplusplus)
extern "C" {
#endif

#include "f77_bindings.h"

//---------------------------------------------------------------------------
//                       Initialize the parameters.
//---------------------------------------------------------------------------

//! Fortran binding.
void 
cpt_init_parameters_f( const Real *max_distance );

//! C binding.
void 
cpt_init_parameters_c( const Real max_distance );


//---------------------------------------------------------------------------
//                       Initialize the grid.
//---------------------------------------------------------------------------

//! Fortran binding.
/*!
  \param compute_grad_distance
  If compute_grad_distance is non-zero, the gradient of the distance will 
  be computed.  If compute_grad_distance is zero, it will not be computed.  
  Then just pass a Real for grad_distance instead of an array.
  \param compute_closest_point
  If compute_closest_point is non-zero, the closest point will be computed.
  If compute_closest_point is zero, it will not be computed.  Then just
  pass a Real for closest_point instead of an array.
  \param compute_closest_face:
  If compute_closest_face is non-zero, the closest face will be computed.
  If compute_closest_face is zero, it will not be computed.  Then just
  pass an integer for closest_face instead of an array.
*/
void 
cpt_init_grid_f( const Real* cartesian_domain, 
		 const int* grid_dimensions,
		 Real* distance,
		 int* compute_grad_distance,
		 Real* grad_distance,
		 int* compute_closest_point,
		 Real* closest_point,
		 int* compute_closest_face,
		 int* closest_face );
  
//! C binding.
void 
cpt_init_grid_c( const Real* cartesian_domain, 
		 const int* grid_dimensions,
		 Real* distance,
		 Real* grad_distance,
		 Real* closest_point,
		 int* closest_face );
  

//---------------------------------------------------------------------------
//                       Initialize the b-rep.
//---------------------------------------------------------------------------

//! Fortran binding
void 
cpt_init_brep_noclip_f( const int *num_vertices,
			const Real* vertices,
			const int *num_faces,
			const int* faces );

//! C binding
void 
cpt_init_brep_noclip_c( const int num_vertices,
			const Real* vertices,
			const int num_faces,
			const int* faces );


//---------------------------------------------------------------------------
//                       Initialize the b-rep.  
//    Clip the mesh to use only points that affect the cartesian domain.
//---------------------------------------------------------------------------

//! Fortran binding
void 
cpt_init_brep_f( const int *num_vertices,
		 const Real* vertices,
		 const int *num_faces,
		 const int* faces );

//! C binding
void 
cpt_init_brep_c( const int num_vertices,
		 const Real* vertices,
		 const int num_faces,
		 const int* faces );


//---------------------------------------------------------------------------
//                      Closest Point Transform.
//---------------------------------------------------------------------------

//! Fortran binding
void 
cpt_f( void );

//! C binding
void 
cpt_c( void );


//---------------------------------------------------------------------------
//    Closest Point Transform using bbox scan conversion algorithm.
//---------------------------------------------------------------------------

//! Fortran binding
void 
cpt_bbox_f( void );

//! C binding
void 
cpt_bbox_c( void );


//---------------------------------------------------------------------------
//    Closest Point Transform using brute force algorithm.
//---------------------------------------------------------------------------

//! Fortran binding
void 
cpt_brute_f( void );

//! C binding
void 
cpt_brute_c( void );


//---------------------------------------------------------------------------
//    Flood fill the distance.
//---------------------------------------------------------------------------

//! Fortran binding
int 
cpt_flood_fill_distance_f( const Real far_away );

//! C binding
int 
cpt_flood_fill_distance_c( const Real far_away );


//---------------------------------------------------------------------------
//    Check if the grids are valid.
//---------------------------------------------------------------------------

//! Fortran binding
int 
cpt_valid_grid_f( void );

//! C binding
int 
cpt_valid_grid_c( void );


//---------------------------------------------------------------------------
//              Find the sign of the distance in the processors.
//---------------------------------------------------------------------------

//! Fortran binding
int 
cpt_find_sign_distance_f( const int* process_dimensions,
			const int* process,
			const int* sign_distance );

//! C binding
int 
cpt_find_sign_distance_c( const int* process_dimensions,
			  const int* process,
			  const int* sign_distance );


//---------------------------------------------------------------------------
//                    Check if a surface is oriented.
//---------------------------------------------------------------------------

//! Fortran binding
int 
oriented_surface_f( const int* num_faces, const int* faces );

//! C binding
int 
oriented_surface_c( const int num_faces, const int* faces );


//---------------------------------------------------------------------------
//                            Orient a surface.
//---------------------------------------------------------------------------

//! Fortran binding
void 
orient_surface_f( const int* num_faces, int* faces );

//! C binding
void 
orient_surface_c( const int num_faces, int* faces );


//---------------------------------------------------------------------------
//        Write information about the state of the grid variables.
//---------------------------------------------------------------------------

//! Fortran binding
void 
cpt_write_grid_state_f( void );

//! C binding
void 
cpt_write_grid_state_c( void );


//---------------------------------------------------------------------------
//        Write information about the state of the brep variables.
//---------------------------------------------------------------------------

//! Fortran binding
void 
cpt_write_brep_state_f( void );

//! C binding
void 
cpt_write_brep_state_c( void );


//---------------------------------------------------------------------------
//        Write information about the state of the variables.
//---------------------------------------------------------------------------

//! Fortran binding
void 
cpt_write_state_f( void );

//! C binding
void 
cpt_write_state_c( void );


//---------------------------------------------------------------------------
//          Write information about the state of the grid.
//---------------------------------------------------------------------------

//! Fortran binding
void 
cpt_check_grid_f( void );

//! C binding
void 
cpt_check_grid_c( void );


//---------------------------------------------------------------------------
//          Write information about the state of the grid.
//---------------------------------------------------------------------------

//! Fortran binding
void 
cpt_check_mesh_c( int num_vertices, Real* vertices, 
		  int num_faces, int* faces );

//! C binding
void 
cpt_check_mesh_f( int num_vertices, Real* vertices, 
		  int num_faces, int* faces );


//---------------------------------------------------------------------------
//                          Write the mesh.
//---------------------------------------------------------------------------

//! Fortran binding
void 
cpt_write_mesh_f( void );

//! C binding
void 
cpt_write_mesh_c( void );


//---------------------------------------------------------------------------
//                           Array transpose.
//---------------------------------------------------------------------------

//! Fortran binding
void 
cpt_transpose_int_f( int* xdim, int* ydim, int* array );

//! C binding
void 
cpt_transpose_int_c( int xdim, int ydim, int* array );


//! Fortran binding
void 
cpt_transpose_real_f( int* xdim, int* ydim, Real* array );

//! C binding
void 
cpt_transpose_real_c( int xdim, int ydim, Real* array );

#if defined(__cplusplus)
}
#endif

// End of file



