// -*- C++ -*-
//----------------------------------------------------------------------
//
//                            Sean Mauch
//                California Institute of Technology
//                   (C) 1999 All Rights Reserved
//
//----------------------------------------------------------------------

// $Log: Grid.ipp,v $
// Revision 1.11  2000/09/17 08:22:19  sean
// Added the gradient of the distance to the computed arrays.
//
// Revision 1.10  2000/07/28 19:38:33  sean
// Moved includes from .h to .cc.
// Added cart_epsilon and grid_epsilon.
//
// Revision 1.9  2000/06/30 23:39:53  sean
// Do cart_to_grid() as the polyhedra are made.
//
// Revision 1.8  2000/06/30 20:38:08  sean
// Improved grid_to_cart() and distance_closest_point()
//
// Revision 1.7  2000/06/26 08:24:31  sean
// Now return const references instead of objects for accessors.
//
// Revision 1.6  2000/04/18 07:33:20  sean
// Fixed set_far_away().  Added grid_size(void).
//
// Revision 1.5  2000/04/05 11:20:59  sean
// Added distance_closest_point_checked().
//
// Revision 1.4  2000/03/20 16:00:53  sean
// *** empty log message ***
//
// Revision 1.3  2000/03/08 11:44:30  sean
// Made set_far_away more efficient.  Changed comments.
//
// Revision 1.2  2000/02/24 08:56:23  sean
// Added closest face capability.
//
// Revision 1.1  1999/11/28 06:25:13  sean
// Split code for classes into .h, .ipp and .cc files.
//

#if !defined(__Grid_inl__)
#error This file is an implementation detail of the class Grid
#endif

// Inline method definitions

//------------------------------Accesors-----------------------------------

inline
Real 
Grid::distance( const int i, const int j, const int k ) const 
{
  // assert( 0 <= i && i < _grid_size[0] &&
  //         0 <= j && j < _grid_size[1] &&
  //         0 <= k && k < _grid_size[2] );
  return _distance[ i + j * _y_stride + k * _z_stride ];
}

inline
Real& 
Grid::distance( const int i, const int j, const int k ) 
{
  // assert( 0 <= i && i < _grid_size[0] &&
  //         0 <= j && j < _grid_size[1] &&
  //         0 <= k && k < _grid_size[2] );
  return _distance[ i + j * _y_stride + k * _z_stride ];
}

inline
int 
Grid::compute_grad_distance( void ) const
{
  return _compute_grad_distance;
}

inline
Point 
Grid::grad_distance( const int i, const int j, const int k ) const 
{
  // assert( 0 <= i && i < _grid_size[0] &&
  //         0 <= j && j < _grid_size[1] &&
  //         0 <= k && k < _grid_size[2] );
  Real* rptr = _grad_distance + 3 * ( i + j * _y_stride + k * _z_stride );
  Real x = *(rptr++);
  Real y = *(rptr++);
  Real z = *rptr;
  return Point( x, y, z );
}

inline
void 
Grid::set_grad_distance( const int i, const int j, const int k, 
			 const Point& pt ) 
{
  // assert( 0 <= i && i < _grid_size[0] &&
  //         0 <= j && j < _grid_size[1] &&
  //         0 <= k && k < _grid_size[2] );
  Real* rptr = _grad_distance + 3 * ( i + j * _y_stride + k * _z_stride );
  *(rptr++) = pt.x();
  *(rptr++) = pt.y();
  *rptr     = pt.z();
}


inline
int 
Grid::compute_closest_point( void ) const
{
  return _compute_closest_point;
}


inline
Point 
Grid::closest_point( const int i, const int j, const int k ) const 
{
  // assert( 0 <= i && i < _grid_size[0] &&
  //         0 <= j && j < _grid_size[1] &&
  //         0 <= k && k < _grid_size[2] );
  Real* rptr = _closest_point + 3 * ( i + j * _y_stride + k * _z_stride );
  Real x = *(rptr++);
  Real y = *(rptr++);
  Real z = *rptr;
  return Point( x, y, z );
}

inline
void 
Grid::set_closest_point( const int i, const int j, const int k, 
			const Point& pt ) 
{
  // assert( 0 <= i && i < _grid_size[0] &&
  //         0 <= j && j < _grid_size[1] &&
  //         0 <= k && k < _grid_size[2] );
  Real* rptr = _closest_point + 3 * ( i + j * _y_stride + k * _z_stride );
  *(rptr++) = pt.x();
  *(rptr++) = pt.y();
  *rptr     = pt.z();
}

inline
int 
Grid::compute_closest_face( void ) const
{
  return _compute_closest_face;
}

inline
int 
Grid::closest_face( const int i, const int j, const int k ) const 
{
  // assert( 0 <= i && i < _grid_size[0] &&
  //         0 <= j && j < _grid_size[1] &&
  //         0 <= k && k < _grid_size[2] );
  return _closest_face[i + j * _y_stride + k * _z_stride];
}

inline
int&
Grid::closest_face( const int i, const int j, const int k )
{
  // assert( 0 <= i && i < _grid_size[0] &&
  //         0 <= j && j < _grid_size[1] &&
  //         0 <= k && k < _grid_size[2] );
  return _closest_face[i + j * _y_stride + k * _z_stride];
}

inline
int 
Grid::grid_size( int d ) const
{
  // assert( 0 <= d < 3 );
  return _grid_size[d];
}

inline
int 
Grid::grid_size( void ) const
{
  // assert( 0 <= d < 3 );
  return _grid_size[0] * _grid_size[1] * _grid_size[2];
}

inline
const Point&
Grid::min_point( void ) const
{
  return _min_point;
}

inline
const Point&
Grid::max_point( void ) const
{
  return _max_point;
}

inline
const Real&
Grid::delta( int i ) const
{
  return _delta[i];
}

// Change from Grid coordinates to Cartesian coordinates
inline
void
Grid::grid_to_cart( const int i, const int j, const int k, Point& p ) const 
{
  //assert( 0 <= i && i < _grid_size[0] && 
  //        0 <= j && j < _grid_size[1] && 
  //        0 <= k && k < _grid_size[2] );
  p.x() = _min_point.x() + i * _delta[0];
  p.y() = _min_point.y() + j * _delta[1];
  p.z() = _min_point.z() + k * _delta[2];
}

inline
void
Grid::cart_to_grid( Point& p ) const
{
  p.x() = (p.x() - _min_point.x()) / _delta[0];
  p.y() = (p.y() - _min_point.y()) / _delta[1];
  p.z() = (p.z() - _min_point.z()) / _delta[2];
}

inline
void
Grid::vector_cart_to_grid( Point& p ) const
{
  p.x() /= _delta[0];
  p.y() /= _delta[1];
  p.z() /= _delta[2];
}

inline
Real 
Grid::grid_epsilon() const
{
  return _grid_epsilon;
}

inline
Real 
Grid::cart_epsilon() const
{
  return _cart_epsilon;
}

//
// End of file
