// -*- C++ -*-
//----------------------------------------------------------------------
//
//                            Sean Mauch
//                California Institute of Technology
//                   (C) 1999 All Rights Reserved
//
//----------------------------------------------------------------------

// $Log: Grid.h,v $
// Revision 1.16  2001/01/07 03:15:51  sean
// Made HUGE stand for unknown.
//
// Revision 1.15  2000/09/17 08:22:19  sean
// Added the gradient of the distance to the computed arrays.
//
// Revision 1.14  2000/07/28 19:38:33  sean
// Moved includes from .h to .cc.
// Added cart_epsilon and grid_epsilon.
//
// Revision 1.13  2000/06/30 23:39:53  sean
// Do cart_to_grid() as the polyhedra are made.
//
// Revision 1.12  2000/06/30 20:38:08  sean
// Improved grid_to_cart() and distance_closest_point()
//
// Revision 1.11  2000/06/26 08:24:31  sean
// Now return const references instead of objects for accessors.
//
// Revision 1.10  2000/06/16 22:37:13  sean
// Moved includes from Grid.h to Grid.cc.
//
// Revision 1.9  2000/04/18 07:33:20  sean
// Fixed set_far_away().  Added grid_size(void).
//
// Revision 1.8  2000/04/05 21:33:12  sean
// Changed iostream.h to iostream.
//
// Revision 1.7  2000/04/05 11:20:59  sean
// Added distance_closest_point_checked().
//
// Revision 1.6  2000/03/20 16:00:53  sean
// *** empty log message ***
//
// Revision 1.5  2000/03/09 11:10:12  sean
// Took out inside Polyhedron checking to speed up.
//
// Revision 1.4  2000/03/08 11:44:30  sean
// Made set_far_away more efficient.  Changed comments.
//
// Revision 1.3  2000/02/24 08:56:23  sean
// Added closest face capability.
//
// Revision 1.2  1999/12/01 23:27:24  sean
// Added the make() function.  Allow copying.
//
// Revision 1.1  1999/11/28 06:25:13  sean
// Split code for classes into .h, .ipp and .cc files.
//

#if !defined(__Grid_h__)
#define __Grid_h__

#include "cpt_defs.h"

#include <iosfwd>
#include <vector>

#include "Point.h"
#include "Coord.h"

class Vertex;
class Edge;
class Face;

using namespace std;


class Grid
{
public:

  Grid();
  Grid ( int gx, int gy, int gz, const Point& minpt, const Point& maxpt,
         Real* distance, Real* grad_distance, 
	 Real* closest_point, int* closest_face );
  Grid( const Grid& rhs );
  const Grid& operator=( const Grid& rhs );
  
  virtual ~Grid();
  void make ( int gx, int gy, int gz, const Point& minpt, const Point& maxpt,
	      Real* distance, Real* grad_distance,
	      Real* closest_point, int* closest_face );
  
  //------------------------------Accesors-----------------------------------
  Real distance( const int i, const int j, const int k ) const;
  Real& distance( const int i, const int j, const int k ); 
  int compute_grad_distance() const;
  Point grad_distance( const int i, const int j, const int k ) const; 
  void set_grad_distance( const int i, const int j, const int k, 
			  const Point& pt ); 
  int compute_closest_point() const;
  Point closest_point( const int i, const int j, const int k ) const; 
  void set_closest_point( const int i, const int j, const int k, 
			  const Point& pt ); 
  int compute_closest_face() const;
  int closest_face( const int i, const int j, const int k ) const; 
  int& closest_face( const int i, const int j, const int k ); 
  int grid_size( int d ) const;
  int grid_size() const;
  const Point& min_point() const;
  const Point& max_point() const;
  const Real& delta( int i ) const;
  Real grid_epsilon() const;
  Real cart_epsilon() const;
  
  //-------------------------Mathematical operations--------------------------
  void grid_to_cart( const int i, const int j, const int k, Point& p ) const; 
  void cart_to_grid( Point& p ) const;
  void vector_cart_to_grid( Point& p ) const;
  int distance_closest_point( const vector<Coord>& cs, const Vertex& vert,
			      const Real max_distance );
  int distance_closest_point( const vector<Coord>& cs, const Edge& edge,
			      const Real max_distance );
  int distance_closest_point( const vector<Coord>& cs, const Face& face,
			      const Real max_distance );
  int distance_closest_point_checked( const vector<Coord>& cs, 
				      const Vertex& vert, 
				      const Real max_distance );
  int distance_closest_point_checked( const vector<Coord>& cs, 
				      const Edge& edge, 
				      const Real max_distance );
  int distance_closest_point_checked( const vector<Coord>& cs, 
				      const Face& face, 
				      const Real max_distance );
  void initialize();
  void set_far_away();
  int flood_fill( const Real );

protected:
  
private:

  int _grid_size[3];			// Size of the grid
  int _y_stride, _z_stride, _i_stride;
  Point _min_point, _max_point;	        // The corners of the box
  Real _length[3];			// Sides of the box
  Real _delta[3];			// Grid spacing
  Real _grid_epsilon;
  Real _cart_epsilon;
  Real* _distance;                  	// Array of distances.
  Real* _grad_distance;                	// Array of grad distance.
  int _compute_grad_distance;           // Whether to compute grad dist.
  Real* _closest_point;               	// Array of closest Points.
  int _compute_closest_point;           // Whether to compute c.p.
  int* _closest_face;                 	// Array of closest faces.
  int _compute_closest_face;            // Whether to compute closest face

  void copy( const Grid& g );
  int distance( const vector<Coord>& cs, const Vertex& vert );
  int distance( const vector<Coord>& cs, const Edge& edge );
  int distance( const vector<Coord>& cs, const Face& face, Real delta );
  int closest_point( const vector<Coord>& cs, const Vertex& vert );
  int closest_point( const vector<Coord>& cs, const Edge& edge );
  int closest_point( const vector<Coord>& cs, const Face& face, Real delta );

};

//-------------------------------File IO--------------------------------------
//ostream& operator << ( ostream& out, const Grid& g );

#define __Grid_inl__
#include "Grid.inl"
#undef __Grid_inl__

#endif

//
// End of file
