// -*- C++ -*-
//----------------------------------------------------------------------
//
//                            Sean Mauch
//                California Institute of Technology
//                   (C) 1999 All Rights Reserved
//
//----------------------------------------------------------------------

// $Log: BRep.h,v $
// Revision 1.16  2001/01/07 03:15:51  sean
// Made HUGE stand for unknown.
//
// Revision 1.15  2000/07/12 00:24:34  sean
// Added face_id to fix wrong face index.
//
// Revision 1.14  2000/06/26 08:24:31  sean
// Now return const references instead of objects for accessors.
//
// Revision 1.13  2000/06/22 22:52:08  sean
// Used #ifdef CPT_BREP_IO to get rid of some IO.  Perhaps more portable.
//
// Revision 1.12  2000/06/21 23:57:50  sean
// Moved #include <stack> from BRep.h to cpt.cc
//
// Revision 1.11  2000/06/16 22:20:09  sean
// Moved some includes from BRep.h to BRep.cc.
//
// Revision 1.10  2000/04/19 17:51:30  sean
// Added a make() that uses the cartesian domain.
//
// Revision 1.9  2000/04/18 07:29:15  sean
// Added a make() that doesn't use cartesian domain.
//
// Revision 1.8  2000/04/10 18:02:25  aivazis
// MIPSpro-7.3 fixes: includes and (char *) casts
//
// Revision 1.7  2000/04/05 21:33:12  sean
// Changed iostream.h to iostream.
//
// Revision 1.6  2000/04/05 11:15:05  sean
// Added make.  Changed notation for cartesian_domain.
//
// Revision 1.5  2000/03/20 16:00:53  sean
// *** empty log message ***
//
// Revision 1.4  2000/03/08 14:28:03  sean
// Change constructor to a linear time algorithm.
//
// Revision 1.3  2000/02/24 08:50:30  sean
// Added closest face capability.
//
// Revision 1.2  1999/12/30 22:20:12  sean
// Changed the mathematica_print function.
//
// Revision 1.1  1999/11/28 06:25:13  sean
// Split code for classes into .h, .ipp and .cc files.
//

#if !defined(__BRep_h__)
#define __BRep_h__

#include "cpt_defs.h"

#include <iosfwd>
#include <vector>
#include <algorithm>

#include "util.h"
#include "Point.h"
#include "Segment.h"
#include "Vertex.h"
#include "Edge.h"
#include "Face.h"
#include "Polyhedron.h"

using namespace std;

class BRep
{
public:

  BRep();
  BRep( const int num_vertices, const Real* vertices,
	const int num_faces, const int* faces,
	const Real* cartesian_domain,
	const Real max_distance );
  BRep( const BRep& rhs );
  const BRep& operator=( const BRep& rhs );
  
  virtual ~BRep();

  void make( const int num_vertices, const Real* vertices,
	     const int num_faces, const int* faces );
  void make( const int num_vertices, const Real* vertices,
	     const int num_faces, const int* faces,
	     const Real* cartesian_domain,
	     const Real max_distance );
  
  //-----------------------------Accesors-------------------------------------
  int num_vertices() const;
  int num_half_edges() const;
  int num_edges() const;
  int num_faces() const;
  int num_face_id() const;

  const Point& vertex( int index ) const;
  int adjacent_edge( int index ) const;
  vector<int>& edge_list( int index );

  const Double<int>& half_edge( int index ) const;
  int adjacent_face( int index ) const;
  const Triple<int>& face( int index ) const;
  const Point& face_normal( int index ) const;
  void edge_vertices( int edge_index, Point& source, Point& target ) const;
  int face_vertex_index ( int face_index, int vertex_index ) const;
  const Point& face_vertex( int face_index, int vertex_index ) const;
  int edge_adjacent_face_index( const int ) const;
  int edge_adjacent_face_normals( int edge_index, Point& left_normal,
				  Point& right_normal ) const;
  void get_face( int index, Face& face ) const;
  int get_edge( int index, Edge& edge ) const;
  int get_opposite_edge ( int half_edge_index ) const;
  int get_previous_edge ( int face_index, int edge_index ) const;
  int get_vertex( int index, Vertex& vertex ) const; 
  int face_id( int index ) const;

  //-----------------------------Mathematical Operations-----------------------
  Double<int> closest_point( Grid& grid, Real max_distance ) const;
  Double<int> closest_point_bbox( Grid& grid, Real max_distance ) const;

  //-------------------------------IO----------------------------------------
#ifdef CPT_BREP_IO
  void mathematica_print( ostream& out );
  void check( void );
#endif // CPT_BREP_IO

protected:
  
private:

  const int _unknown;  
  vector<Point> _vertices;
  // An adjacent edge to each vertex.
  vector<int> _adjacent_edges; 
  // A list of edges that lead to a higher vertex index.
  vector< vector<int> >* _edge_list;

  // A half edge is determined by the indices of two vertices.
  vector< Double<int> > _half_edges; 
  // The adjacent faces to each half edge.
  vector<int> _adjacent_faces;

  // A face is determined by the indices of three half edges.
  vector< Triple<int> > _faces;
  // The unit outward normals of the faces.
  vector<Point> _face_normals;
  vector<int> _face_id;
  int _num_face_id;

  void copy( const BRep& br );
  void clear ( void );
  void init( int num_vert, int num_face );
  void add_vertex ( const Point& p );
  int add_half_edge ( int source, int target, int face );
  void add_face ( int a, int b, int c );
};

//---------------------------File IO-----------------------------------------
//istream& operator >> ( istream& in, BRep& br );
//ostream& operator << ( ostream& out, const BRep& br );

#define __BRep_inl__
#include "BRep.inl"
#undef __BRep_inl__

#endif

//
// End of file
