///////////////////////////////////////////////////////////////////////////
//                                                                       //
//  Class: CVertex3d                                                     //
//                                                                       //
//  Vertex class containing geometric position, normal, color,           //
//  a list of vertex neighbors, a list of face neighbors, and            //
//  other miscellaneous flags.                                           //
//                                                                       //
///////////////////////////////////////////////////////////////////////////

#ifndef _VERTEX_3D_
#define _VERTEX_3D_

#include "Object3d.h"
#include "Vector3d.h"
#include "Color.h"
#include "Array3d.h"

class CMesh3d;
class CFace3d;
class CEdge3d;

class CVertex3d : public CObject3d
{

private :

  CVector3d m_Coord;	 // Geometry
  CVector3d m_Normal;    // Normal
  CColor    m_Color;     // Color
  char      m_Flag;      // Flag
  void     *m_pBag;		 // Any associated structure
  float     m_u,m_v;     // texture coordinates

  // Neighbors (faces and vertices)
  CArray3d<CVertex3d> m_ArrayVertexNeighbors;
  CArray3d<CFace3d>   m_ArrayFaceNeighbors;

  float *confWeight;
  float *areaWeight;
  float *lengthWeight;

  // added by LHK(05/02/2001)
  int   burnedLevel;

public :

  // Constructors
  CVertex3d() {	burnedLevel = 0; }
  CVertex3d(const float x, const float y, const float z)
    { SetPosition(x,y,z); burnedLevel = 0;}
  CVertex3d(CVertex3d& v) { Set(v); burnedLevel = 0; }
  CVertex3d(CVertex3d *pV) { Set(pV); burnedLevel = 0; }
  CVertex3d(CVertex3d *pVertex, CVector3d *pVector)
    { Set(pVertex); Move(pVector); }

  // added by LHK(05/02/2001)
  int GetBurnedLevel() { return burnedLevel; }
  void SetBurnedLevel(int bl) { burnedLevel = bl; }

  // Destructor (CArray3d will do the work...)
  virtual ~CVertex3d() { }

  // Texture coordinate
  float u() { return m_u; }
  float v() { return m_v; }

  // Data setting
  void Clear() { m_Coord.Set(0.f,0.f,0.f); }
  void SetPosition(const float x, const float y, const float z)
    { m_Coord.Set(x,y,z); }
  void SetPosition(CVector3d& pos) { m_Coord = pos; }
  void Set(CVertex3d *pV) { m_Coord = pV->GetPosition(); }
  void Set(CVertex3d& v)  { m_Coord = v.GetPosition(); }
  void SetUV(float u, float v) { m_u = u; m_v = v;}

  // Moving
  void Move(const float dx, const float dy, const float dz)
    { m_Coord[0] += dx; m_Coord[1] += dy; m_Coord[2] += dz; }
  void Move(CVector3d &vector, const float ratio = 1.0f)
    { m_Coord += ratio * vector; }
  void Move(CVector3d *pVector, const float ratio = 1.0f)
    { m_Coord += ratio * *(pVector); }

  // Per coordinate
  void Set(const unsigned int index, const float value)
    { m_Coord[index] = value;}
  void x(const float x) { m_Coord[0] = x; }
  void y(const float y) { m_Coord[1] = y; }
  void z(const float z) { m_Coord[2] = z; }

  // Data access 
  CVector3d* GetPosition() { return &m_Coord; }
  float Get(const unsigned int index) const { return m_Coord[index]; }
  float x() const { return m_Coord[0];  }
  float y() const { return m_Coord[1];  }
  float z() const { return m_Coord[2]; }

  // Type
  virtual int GetType() const;

  // Color
  CColor* GetColor() { return &m_Color; }
  void SetColor(CColor& color) { m_Color.Set(color); }

  // Flag
  char GetFlag() const { return m_Flag; }
  void SetFlag(const char flag) { m_Flag = flag; }
  int HasNeighborWithFlag(int flag);
  int HasNeighborWithFlagButDiff(int flag,CVertex3d *pVertexDiff);
  CVertex3d* GetFirstVertexNeighborWithFlag(int flag);
  CVertex3d* GetNearestVertexNeighborWithFlag(int flag);
  CVertex3d* GetFirstVertexNeighborWithFlagButDiff(int flag,
						   CVertex3d *pVertexDiff);
  CVertex3d* GetFirstNeighborWithFlagAndSharingFaceWithFlag(int vertexFlag,
							    int faceFlag);
  int DiffFlagsOnNeighboringFaces();
  int IsFlagSmallestOnNeighboringFaces(int flag);
  int FindVertexNeighborsWithGreaterFlag(int flag,CArray3d<CVertex3d> &array);
  int FindVertexNeighborsWithSmallerFlag(int flag,CArray3d<CVertex3d> &array);
  int NumVertexNeighborsWithFlag(int flag);

  // Bag, dirty :-(
  void  SetBag(void *pBag) { m_pBag = pBag; }
  void* GetBag() { return m_pBag; }

  // Normal
  void SetNormal(const float x,const float y,const float z) 
    { m_Normal.Set(x,y,z); }
  void SetNormal(CVector3d &vector) { m_Normal.Set(vector); }
  CVector3d* GetNormal() { return &m_Normal; }

  // Comparison Operators
  int Diff(CVertex3d *pVertex);
  int Equal(CVertex3d *pVertex);
  int NearEqual(CVertex3d *pVertex, float tolerence=0.0001);
  
  // Vertex neighbors
  int  NumVertexNeighbors()
    { return m_ArrayVertexNeighbors.GetSize(); }
  void AddNeighbor(CVertex3d *pVertex);
  void AddNeighborNoCheck(CVertex3d *pVertex);
  int  HasNeighbor(CVertex3d *pVertex)
    { return m_ArrayVertexNeighbors.Has(pVertex); }
  int  RemoveNeighbor(CVertex3d *pVertex);
  void RemoveAllVertexNeighbors();
  int  UpdateNeighbor(CVertex3d *pVertexOld, CVertex3d *pVertexNew);
  void UpdateNeighborRecursive(CVertex3d *pVertexOld, CVertex3d *pVertexNew);
  int  FindFaceAroundContainingVertex(CVertex3d *pVertex,
				      CArray3d<CFace3d> &ArrayFace);
  CVertex3d* GetVertexNeighbor(const unsigned int index) 
    {return m_ArrayVertexNeighbors[index];}
  CArray3d<CVertex3d> *GetArrayVertexNeighbors() 
    { return &m_ArrayVertexNeighbors; }

  // Face neighbors
  int  NumFaceNeighbors()
    { return m_ArrayFaceNeighbors.GetSize(); }
  void AddNeighbor(CFace3d *pFace);
  void AddNeighborNoCheck(CFace3d *pFace);
  int  RemoveNeighbor(CFace3d *pFace);
  void RemoveAllFaceNeighbors();
  int  HasNeighbor(CFace3d *pFace) { return m_ArrayFaceNeighbors.Has(pFace); }
  CFace3d *GetFaceNeighbor(const unsigned int index)
    { return m_ArrayFaceNeighbors[index]; }
  CArray3d<CFace3d> *GetArrayFaceNeighbors()
    { return &m_ArrayFaceNeighbors; }

  // Face sharing
  int FindSharingFaces(CVertex3d *pVertex,CArray3d<CFace3d> &array);

  // Differential Geometry Operators
  CVector3d GetXu();
  CVector3d GetXv();
  CVector3d GetXuv();
  CVector3d GetXvu();
  CVector3d GetUmbrella();

  CVector3d MeanCurvatureNormal();
  double    GaussianCurvature();
  CVector3d Laplacian();
  CVector3d Umbrella();
  float GauB_Curvature();   // added for geodesics...
  float DistanceTo(CVertex3d *pV);

  double GetMeanAreaAround();
  double GetMaxAreaAround();
  double GetMinAreaAround();
  double GetMaxAngleAround();

  double Area1Ring();

  // Weights
  float GetConfWeight(int i)   { return confWeight[i]; }
  float GetAreaWeight(int i)   { return areaWeight[i]; }
  float GetLengthWeight(int i) { return lengthWeight[i]; }
  float GetConfWeight(CVertex3d *p)
	{ return confWeight[m_ArrayVertexNeighbors.IndexFrom(p)]; }
  float GetAreaWeight(CVertex3d *p)
	{ return areaWeight[m_ArrayVertexNeighbors.IndexFrom(p)]; }
  float GetLengthWeight(CVertex3d *p)
	{ return lengthWeight[m_ArrayVertexNeighbors.IndexFrom(p)]; }
  void  CalcWeights();

  // Edges
  double GetMeanEdgeLengthAround();
  int NumSharpEdges(const double threshold = 0.5);

  // Boundaries
  int IsOnBoundary();
 
  // Debug
  void Trace();

  // OpenGL
  virtual int glDraw();
  void glDraw(const float radius,
	      unsigned char *ColorVertex,CMesh3d *pMesh = NULL);
  void glDrawHighlight(const float radius,const float RadiusNeighbor,
		       unsigned char *ColorVertex,CMesh3d *pMesh = NULL,
		       unsigned char *ColorNeightbor = NULL);
};


#endif // _VERTEX_3D_
