///////////////////////////////////////////////////////////////////////////
//                                                                       //
//  Class: CVector2d                                                     //
//                                                                       //
//  2D Vector to represent points or directions.  Each component of the  //
//  vector is stored as a floating point number.                         //
//                                                                       //
///////////////////////////////////////////////////////////////////////////

#ifndef _VECTOR_2D_
#define _VECTOR_2D_

class CVector2d
{

private :

protected:

 float vec[2];  // Storage for the vector components

public :

  // Constructors
  CVector2d() { }
  CVector2d(const float x, const float y)
    { vec[0] = x; vec[1] = y; }
  CVector2d(const float v[2])
    { vec[0] = v[0]; vec[1] = v[1]; }
  CVector2d(const CVector2d &vector)  { Set(vector); }
  CVector2d(const CVector2d *pVector) { Set(pVector); }

  virtual ~CVector2d() { }

  // Debug
  void Trace() const;

  // Data setting
  virtual int GetType() const;
  void Clear()
    { vec[0] = 0.f; vec[1] = 0.f; }
  void Set(const CVector2d *pVector) { Set(pVector->GetArray()); }
  void Set(const CVector2d& vector)  { Set(vector.GetArray()); }
  void Set(const float x, const float y)
    { vec[0] = x; vec[1] = y; }
  void Set(const float v[2])
    { vec[0] = v[0]; vec[1] = v[1]; }

  // Data Access
  const float* GetArray() const { return vec; }
  void         Get(float& x, float& y) const;

  // Per coordinate (explicit inline functions)
  void x(float newX) { vec[0] = newX; }
  void y(float newY) { vec[1] = newY; }

  // Data access (explicit inline functions)
  float x() const { return (vec[0]); }
  float y() const { return (vec[1]); }

  // Data access using indices
  float&       operator[](int i)       { return (vec[i]); }
  const float& operator[](int i) const { return (vec[i]); }

  // Operators
  CVector2d& operator+=(const CVector2d& rVector);
  CVector2d& operator+=(const CVector2d* pVector);
  CVector2d& operator-=(const CVector2d& rVector);
  CVector2d& operator-=(const CVector2d* pVector);
  CVector2d& operator*=(float d);
  CVector2d& operator/=(float d)
    { return *this *= (1.f/d); }

  // Nondestructive unary negation - returns a new vector
  CVector2d  operator -() const;

  // Binary operators
  friend CVector2d operator+(const CVector2d& u, const CVector2d& v);
  friend CVector2d operator-(const CVector2d& u, const CVector2d& v);
  friend CVector2d operator*(float s,            const CVector2d& u);
  friend CVector2d operator*(const CVector2d& u, float s)
    { return s * u; }
  friend CVector2d operator/(const CVector2d& u, float s)
    { return (1.f/s) * u; }
  friend int       operator==(const CVector2d& v1, const CVector2d& v2);
  friend int       operator!=(const CVector2d& v1, const CVector2d& v2)
    { return !(v1 == v2); }

  int Equals(const CVector2d& v, float tolerence) const;

  float            Dot(const CVector2d& v) const;
  float            Dot(const CVector2d* pV) const;

  // Misc
  double Normalize();
  double Normalize(float value);
  double Length() const;
  double LengthSquared() const;
  int IsCollinear(CVector2d *pVector) const;
  int IsCollinear(CVector2d &vector) const;
  void Negate();
};

#endif // _VECTOR_2D_


