///////////////////////////////////////////////////////////////////////////
//                                                                       //
//  Class: CMesh3d                                                       //
//                                                                       //
//  A mesh class containing an array of vertices, and array of faces,    //
//  and binding information.                                             //
//                                                                       //
///////////////////////////////////////////////////////////////////////////

#ifndef _MESH_3D_
#define _MESH_3D_

#include "Array3d.h"
#include "Face3d.h"
#include "Material.h"
#include "Texture.h"
#include "Transform.h"
#include "Vector3d.h"
#include "Vertex3d.h"

#include <stdio.h>
#include <string.h>

class CMesh3d : public CObject3d
{

private :

  // Datas
  CArray3d<CVertex3d> m_ArrayVertex;
  CArray3d<CFace3d>   m_ArrayFace;

  CTransform m_Transform;
  char m_Name[256];

  // Apparence & texture
  CMaterial m_Material; // material
  int m_NormalBinding;  // normals defined at vertices or faces ?
  int m_ColorBinding;   // colors defined at vertices or faces ?

  // Texture
  CTexture m_Texture;
  BOOL m_UseTexture;

  BOOL m_DrawGeo ;

  // OpenGL-specific variables
  unsigned int m_ListOpenGL;
  int m_Modified;
  int m_Show;

public :

  // Constructor
  CMesh3d();
  CMesh3d(CMesh3d& m)  { Set(m); }
  CMesh3d(CMesh3d *pM) { Set(pM); }

  virtual ~CMesh3d();

  // Data Setting
  void Free();
  void Set(CMesh3d& m);
  void Set(CMesh3d *pM);
  
  // Texture
  void UseTexture(BOOL use) { m_UseTexture = use; }
  BOOL GetTextureStatus() { return m_UseTexture; }
  CTexture *GetTexture() { return &m_Texture; }

  // General Queries
  virtual int GetType() const;
  int IsValid();

  // Faces
  int NumFaces() { return m_ArrayFace.GetSize(); }
  CFace3d *GetFace(int index) { return m_ArrayFace[index]; }
  void AddFace(CFace3d *pFace) { m_ArrayFace.Add(pFace); }
  void SetNumFaces(int NbFace) { m_ArrayFace.SetSize(NbFace); }
  CArray3d<CFace3d> *GetArrayFace() { return &m_ArrayFace; }
  int Has(CFace3d *pFace) { return m_ArrayFace.Has(pFace); }
  int DeleteFace(CFace3d *pFace);
  int DeleteFace(int index);
  int DeleteFacesWithNumNeighbors(int NbNeighbor);
  int FindFace(CFace3d *pFace); // even in neighboring faces
  int IndexFrom(CFace3d *pFace);

  // Vertices
  int NumVertices() { return m_ArrayVertex.GetSize(); }
  CVertex3d *GetVertex(int index) {return m_ArrayVertex[index];}
  void AddVertex(CVertex3d *pVertex) { m_ArrayVertex.Add(pVertex); }
  void SetNumVertices(int NbVertex)  { m_ArrayVertex.SetSize(NbVertex); }
  CArray3d<CVertex3d> *GetArrayVertex() { return &m_ArrayVertex; }
  int Has(CVertex3d *pVertex) { return m_ArrayVertex.Has(pVertex); }
  int DeleteVertex(CVertex3d *pVertex);
  int DeleteVertex(int index);
  int FindVertex(CVertex3d *pVertex);
  int IndexFrom(CVertex3d *pVertex);
  int NumBoundaryVertices();
  void ReorderVertices(); // reorders the boundary vertices to be last

  // Flags
  void SetFlagOnFaces(int flag);
  void SetFlagOnVertices(int flag);
  int GetMaxFlagOnFaces();

  // Normals
  int CalculateNormalPerVertex();
  int CalculateNormalPerFace();
  void SetNormalBinding(int type);
  int  GetNormalBinding() const;

  // Color
  void SetColorBinding(int type);
  int  GetColorBinding();
  void SetFaceColor(CColor& color);

  // added by LHK(04/25/2001)
  void SetFaceColor(CColor &color, int index);

  void SetVertexColor(CColor& color);

  // Material
  CMaterial* GetMaterial() { return &m_Material; }
  void SetMaterial(CMaterial *pMaterial) { m_Material.Set(pMaterial); }

  // Name
  char* GetName() { return m_Name; } 
  void SetName(char *newName) { strcpy(m_Name,newName); } 

  // Transform
  void SetTransform(CTransform &transform) { m_Transform.Set(transform); }
  CTransform* GetTransform() { return &m_Transform; }

  // Range
  void GetRange(int coord,float *min,float *max); 
  void SetRange(int coord,float min,float max); 
  void Offset(int coord,float offset);
  void Scale(int coord,float scale);
  void Move(float dx,float dy,float dz);
  void Move(CVector3d& dX);
  void GetBoundingBox(float *xMin, float *xMax,
		      float *yMin, float *yMax,
		      float *zMin, float *zMax);
  void GetBoundingBox(CVector3d& minBound, CVector3d& maxBound);
  CVector3d GetCenter();
  CVector3d GetCentroid();

  // Modif
  void SetModified(int flag = 1) { m_Modified = flag; }
  int GetModified() const { return m_Modified; }

  // I/O
  int ReadInventor(FILE *fp);
  int ReadVRML(FILE *fp);
  int ReadSMF(FILE *fp);
  int WriteInventor(FILE *fp);
  int WriteVRML(FILE *fp);
  int WriteSMF(FILE *fp);

  // Predefined Meshes
  int GenerateMap(int line,int col,float min,float max);
  int GenerateMap(CTexture *pTexture,int width,int height,int FlagColor = 1);
  int GenerateBox(float dx,float dy,float dz);

  // Adjacency
  int BuildAdjacency();
  void Rebuild();

  // Edges
  float GetMeanEdgeLength();
  int ColorSharpEdges(float threshold,CColor &color);

  // Subdivision
  float LoopAlpha(int n);
  int   SubdivisionLoop(int MoveOnBundary = 1);   // Charles Loop (1987)
  int   SubdivisionDM(int MoveOnBundary = 1);     // Mathieu Desbrun (2001)
  int   SubdivisionSqrt3(int MoveOnBundary = 1);  // Kobbelt (2000)
  int   SubdivisionLinear();

  // Smoothing
  int SmoothLoop(int MoveOnBundary = 1);
  void  CurvatureFlow(float timeStep = .1f);

  // Coloring (curvature, compacity, etc...)
  void ColorHingeAngle();
  void ColorCompacity();
  void ColorHeight();
  void ColorMeanCurvature(double clampLo, double clampHi);
  void ColorGaussianCurvature(double clampLo, double clampHi);

  // Intersection
  int NearestIntersectionWithLine(CVertex3d *pV0,CVertex3d *pV1,
				  CVertex3d *pVertexResult,
				  CFace3d **ppFaceResult,int *pNbFaceVisited);
  // Area
  double GetMinArea(CFace3d **ppFace = NULL);
  double GetMeanArea();

  // Debug
  void Trace();

  // Weights
  void CalcWeights();

  // OpenGL
  virtual int glBuildList();
  virtual int glDraw();
  void    Show(int flag) { m_Show = flag; }
};


#endif // _MESH_3D_
