///////////////////////////////////////////////////////////////////////////
//                                                                       //
//  Class: CMatrix44                                                     //
//                                                                       //
//  4x4 matrix to represent transformations.                             //
//                                                                       //
///////////////////////////////////////////////////////////////////////////

#ifndef _MATRIX_44_
#define _MATRIX_44_

typedef float CMat44[4][4];

#include "Vector3d.h"

class CMatrix44
{

private :

  // Data
  CMat44 m_data;

public :

  // Constructors
  CMatrix44() { }
  CMatrix44(const CMat44& m);
  CMatrix44(const float *data);
  CMatrix44(float a11, float a12, float a13, float a14,
	    float a21, float a22, float a23, float a24,
	    float a31, float a32, float a33, float a34,
	    float a41, float a42, float a43, float a44);
  CMatrix44(const CMatrix44 &rMatrix);
  CMatrix44(const CMatrix44 *pMatrix);

  virtual ~CMatrix44() { }

  // Debug
  void Trace() const;

  // Data setting
  virtual int GetType() const;
  void Clear();
  void MakeIdentity();
  void Set(const CMat44& m);
  void Set(const float *data);
  void Set(const CMatrix44 &rMatrix);
  void Set(const CMatrix44 *pMatrix);

  void SetTranslate(float tx, float ty, float tz);
  void SetRotate(float ax, float ay, float az, float radAngle);
  void SetScale(float sx, float sy, float sz);

  // Per element (explicit inline functions)
  void Set(int i, int j, float data) { m_data[i][j] = data; }

  // Make it look like a usual matrix (so you can do m[3][2])
  float*       operator[](int i)       { return &m_data[i][0]; }
  const float* operator[](int i) const { return &m_data[i][0]; }

  // Data access (explicit inline functions)
  float  Get(int i, int j) const { return m_data[i][j]; }
  float* GetData()               { return &m_data[0][0]; }

  // Operators
  CMatrix44& operator+=(const CMatrix44& rMatrix);
  CMatrix44& operator+=(const CMatrix44* pMatrix);
  CMatrix44& operator-=(const CMatrix44& rMatrix);
  CMatrix44& operator-=(const CMatrix44* pMatrix);
  CMatrix44& operator*=(const float d);
  CMatrix44& operator/=(const float d)
    { return *this *= (1.f/d); }

  // Binary operators
  friend CMatrix44 operator+(const CMatrix44& u, const CMatrix44& v);
  friend CMatrix44 operator-(const CMatrix44& u, const CMatrix44& v);
  friend CMatrix44 operator*(const float s,      const CMatrix44& u);
  friend CMatrix44 operator*(const CMatrix44& u, const float s)
    { return s * u; }
  friend CMatrix44 operator/(const CMatrix44& u, const float s)
    { return (1.f/s) * u; }

  friend CVector3d operator*(const CMatrix44& m, const CVector3d& v)
    { return m.MultMatVec(v); }
  friend CVector3d operator*(const CVector3d& v, const CMatrix44& m)
    { return m.MultVecMat(v); }
  friend CMatrix44 operator*(const CMatrix44& u, const CMatrix44& v)
     { return u.MultLeft(v); }

  // Math
  CVector3d MultMatVec(const CVector3d& v) const;
  CVector3d MultVecMat(const CVector3d& v) const;
  CVector3d MultMatDir(const CVector3d& v) const;
  CVector3d MultDirMat(const CVector3d& v) const;
  CMatrix44 MultLeft(const CMatrix44& mat) const;
  CMatrix44 MultRight(const CMatrix44& mat) const;

  // Misc
  float Determinant() const { return Det4(); }
  float Det4() const;
  float Det3(int r1, int r2, int r3, int c1, int c2, int c3) const;
  float Det2(int r1, int r2, int c1, int c2) const
   { return (m_data[r1][c1]*m_data[r2][c2] - m_data[r2][c1]*m_data[r1][c2]); }
  CMatrix44 Transpose() const;
  CMatrix44 Adjoint() const;
  CMatrix44 Inverse() const;

  static CMatrix44  Identity();
};


#endif // _MATRIX_44_
