///////////////////////////////////////////////////////////////////////////
//                                                                       //
//  Class: CFace3d                                                       //
//                                                                       //
//  Face class containing 3 (+3) vertex pointers, 3 face neighbor        //
//  pointers, a normal vector, a color, a material reference, and        //
//  a flag.                                                              //
//                                                                       //
///////////////////////////////////////////////////////////////////////////

#ifndef _FACE_3D_
#define _FACE_3D_

#include "Array3d.h"
#include "Object3d.h"
#include "Vector3d.h"
#include "Color.h"
#include "Edge3d.h"

class CVertex3d;
class CMesh3d;

class CFace3d : public CObject3d
{

private :

  CVertex3d*   m_pVertex[6];    // Vertices (subdivision included)
  CFace3d*     m_pFace[3];      // Neighbors
  CVector3d    m_Normal;        // Normal
  CColor       m_Color;         // Color
  unsigned int m_IndexMaterial; // Material
  int          m_Flag;          // Flag

  // added by LHK(04/19/2001)
  bool		   m_burned;
  int		   m_NumInOut;
  double	   m_TimeStamp;
  int          m_visitID;

public :

  // Constructors
  CFace3d() { Clear(); }
  CFace3d(CVertex3d *pVertex0,CVertex3d *pVertex1,CVertex3d *pVertex2);
  CFace3d(CFace3d& f)  { Set(f); }
  CFace3d(CFace3d *pF) { Set(pF); }

  virtual ~CFace3d() { }

  // General
  int  IsValid();
  int  Equal(CFace3d *pFace);

  // added by LHK(04/19/2001)
  bool IsBurned() { return m_burned; }
  void SetBurned( bool b ) { m_burned = b; }
  void InFace() { m_NumInOut++; }
  void OutFace() { m_NumInOut--; }
  int  GetNumInOut() { return m_NumInOut; }
  void SetTimeStamp(double time) { m_TimeStamp = time; }
  double GetTimeStamp() { return m_TimeStamp; }
  void SetVisitID( int id ) { m_visitID = id; }
  int  GetVisitID() { return m_visitID; }

  // Data Setting
  void Clear();
  void Set(CVertex3d *pVertex0, CVertex3d *pVertex1, CVertex3d *pVertex2)
    { m_pVertex[0] = pVertex0;
      m_pVertex[1] = pVertex1;
      m_pVertex[2] = pVertex2;   
	}
  void Set(CFace3d *pFace0, CFace3d *pFace1, CFace3d *pFace2)
    { m_pFace[0] = pFace0;
      m_pFace[1] = pFace1;
      m_pFace[2] = pFace2; 	  
  }
  void Set(CVertex3d *pVertex0, CVertex3d *pVertex1, CVertex3d *pVertex2,
	   CFace3d *pFace0, CFace3d *pFace1, CFace3d *pFace2)
    { Set(pVertex0,pVertex1,pVertex2); Set(pFace0,pFace1,pFace2); }
  void Set(CFace3d& f)
    { 
	  Set(f.v0(),f.v1(),f.v2()); Set(f.f0(),f.f1(),f.f2()); 

      // added by LHK(04/19/2001)
	  m_burned = false;
	  m_NumInOut = 0;
	  m_TimeStamp = 0.0;
	  m_visitID = 0;
	}
  void Set(CFace3d *pF)
    { 
	  Set(pF->v0(),pF->v1(),pF->v2()); Set(pF->f0(),pF->f1(),pF->f2()); 

      // added by LHK(04/19/2001)
	  m_burned = false;
	  m_NumInOut = 0;
	  m_TimeStamp = 0.0;
	  m_visitID = 0;
	}

  // Miscellaneous Processing
  CVertex3d* GetCenter();
  CVertex3d* FindNearestVertex(CVertex3d *pVertex);

  // Added by LHK(05/02/2001)
  CVertex3d* FindNearestVertex(CVertex3d *pVertex, float &dist);

  // Neighboring
  int BuildArrayNeighbor(CArray3d<CFace3d> *pArrayFace,int depth = 1);

  // Color
  CColor* GetColor() { return &m_Color; }
  void SetColor(const CColor &c) { m_Color.Set(c); }
  void SetColor(const CColor *pC) { m_Color.Set(pC); }

  // Flag
  int  GetFlag() const { return m_Flag; }
  void SetFlag(int flag) { m_Flag = flag; }
  void SetFlagOnVerticesIfDiff(int flagDiff, int flag);

  // Per vertex
  void v0(CVertex3d *pVertex) { m_pVertex[0] = pVertex; }
  void v1(CVertex3d *pVertex) { m_pVertex[1] = pVertex; }
  void v2(CVertex3d *pVertex) { m_pVertex[2] = pVertex; }
  void v(int index, CVertex3d *pVertex) { m_pVertex[index%6] = pVertex; }
  int  IndexFrom(CVertex3d *pVertex) const;

  // Per neighboring face
  void f0(CFace3d *pFace) { m_pFace[0] = pFace; }
  void f1(CFace3d *pFace) { m_pFace[1] = pFace; }
  void f2(CFace3d *pFace) { m_pFace[2] = pFace; }
  void f(int index, CFace3d *pFace) { m_pFace[index%3] = pFace; }
  int  IndexFrom(CFace3d *pFace) const;

  // Normal
  void SetNormal(float x,float y,float z) { m_Normal.Set(x,y,z); }
  void SetNormal(const CVector3d& n) { m_Normal.Set(n); }
  void SetNormal(const CVector3d *pN) { m_Normal.Set(pN); }
  CVector3d* GetNormal() { return &m_Normal; }
  int HasNormal() { return (m_Normal.LengthSquared() != 0);}
  void CalculateNormal();

  // Sharp edges
  void ColorSharpEdges(double threshold,CColor &color);
  int  GetSharpEdges(double threshold,int *SharpEdge);
  int  HasSharpEdges(double threshold);

  // Type
  virtual int GetType() const;

  // Numbers
  int NumVertices() const;
  int NumFaceNeighbors() const;

  // Per vertex
  CVertex3d* v0() { return m_pVertex[0]; }
  CVertex3d* v1() { return m_pVertex[1]; }
  CVertex3d* v2() { return m_pVertex[2]; }
  CVertex3d* v(int index) { return m_pVertex[index%6]; }

  // Per face
  CFace3d* f0() { return m_pFace[0]; }
  CFace3d* f1() { return m_pFace[1]; }
  CFace3d* f2() { return m_pFace[2]; }
  CFace3d* f(int index) { return m_pFace[index%3]; }

  // Find whenever face contain pVertex
  int HasVertex(CVertex3d *pVertex) const;
  int HasVertex(CVertex3d *pVertex,int *index) const;
  int HasVertexWithFlag(int flag) const;

  int HasSameVertexValue(CVertex3d* pVert);
  CFace3d* NeighborWithTwoVertices(CVertex3d *pVert0, CVertex3d *pVert1) ;
  CFace3d* NeighborWithTwoSameVerticeValues(CVertex3d *pVert0, CVertex3d *pVert1) ;
  float IntersectWithEdge(CEdge3d* eg0, CEdge3d* eg1); // return number of edge of the face

  // Find whenever face contain pFace
  int HasNeighbor(CFace3d *pFace) const;
  int HasNeighbor(CFace3d *pFace,int *index) const;

  // Get vertices and faces (subject to conditions)
  CFace3d* GetOtherNeighbor(CFace3d *pF0,CFace3d *pF1);
  CFace3d* GetNeighborExclusive(CVertex3d *pVertexHas,CVertex3d *pVertexHasNot);
  CFace3d* GetNeighborAcross(CVertex3d *pVertex);
  CVertex3d* GetOtherVertex(CVertex3d *pV0,CVertex3d *pV1);
  CVertex3d* GetVertexExclusive(CVertex3d *pV,CFace3d *pFace);
  CVertex3d* GetVertexAcrossEdge(unsigned int index);
  CVertex3d* GetVertexAcrossEdge(CVertex3d *pVertex);
  void       GetOtherVertices(CVertex3d *&p,CVertex3d *&a, CVertex3d *&b);

  // Join 2 neighboring faces together
  int JoinNeighbors(CFace3d **pFace0,CFace3d **pFace1);

  // Vertex sharing
  int Share1Vertex(CFace3d *pFace);
  int Share2Vertex(CFace3d *pFace);
  int Share2Vertex(CFace3d *pFace,int *IndexEdgeThis,int *IndexEdgeOther);
  int Share2Vertex(CFace3d *pFace,CVertex3d **pSharedV1,CVertex3d **pSharedV2);

  // Find whenever face contain pOld, and update it to pNew
  int UpdateVertex(CVertex3d *pOld,CVertex3d *pNew);
  int UpdateNeighbor(CFace3d *pOld,CFace3d *pNew);
  int UpdateVertexRecursive(CVertex3d *pVertexOld,CVertex3d *pVertexNew);

  // Debug
  void Trace() const;

  // Geometric quantities
  double Area(); 
  double Perimeter();
  double Compacity();
  double Cotangent(CVertex3d *pVertex);
  double Cosine(CVertex3d *pVertex);
  double MixedArea(CVertex3d *pVertex);

  // OpenGL
  void glDraw(unsigned char *ColorFace,
	      CMesh3d *pMesh = NULL,
	      unsigned char *ColorNeightbor = NULL);
};

#endif // _FACE_3D_
